!------------------------------------------------------------------------!
!  The Community Multiscale Air Quality (CMAQ) system software is in     !
!  continuous development by various groups and is based on information  !
!  from these groups: Federal Government employees, contractors working  !
!  within a United States Government contract, and non-Federal sources   !
!  including research institutions.  These groups give the Government    !
!  permission to use, prepare derivative works of, and distribute copies !
!  of their work in the CMAQ system to the public and to permit others   !
!  to do so.  The United States Environmental Protection Agency          !
!  therefore grants similar permission to use the CMAQ system software,  !
!  but users are requested to provide copies of derivative works or      !
!  products designed to operate in the CMAQ system to the United States  !
!  Government without restrictions as to use by others.  Software        !
!  that is used with the CMAQ system but distributed under the GNU       !
!  General Public License or the GNU Lesser General Public License is    !
!  subject to their copyright restrictions.                              !
!------------------------------------------------------------------------!

C RCS file, release, date & time of last delta, author, state, [and locker]
C $Header: /project/work/rep/arc/CCTM/src/biog/beis3/hrno.F,v 1.7 2012/05/09 12:41:30 yoj Exp $

C what(1) key, module and SID; SCCS file; date and time of last delta:
C %W% %P% %G% %U%
      MODULE BDSNP_MOD
      
      IMPLICIT NONE
      REAL,    ALLOCATABLE, SAVE :: NDEPRATE  ( :,: )  ! "ng N m-2 s-1" deposition rate used to update NDEPRES
      LOGICAL, SAVE     :: BDSNP
      LOGICAL, SAVE     :: EPIC      
      
      CONTAINS
C::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
      SUBROUTINE HRNOBDSNP( JDATE, JTIME, TSTEP, NC, NR, COSZEN,
     &            TASFC, SSOLAR, PRES, EMPOL )

C-----------------------------------------------------------------------
C Description:
!  Original BDSNP model code:
!    by  Rynda Hudman, Neil Moore, and Randall Martin 
C  Implemented into CMAQ v 5.0.2 by Quazi Rasool, Rui Zhang, Benjamin Lash   
! !REMARKS:
!  The soil NOx code has been updated from the original implementation
!  of Yienger & Levy [1995] from  Wang et al., [1998] as summarized below.
!  
!  Old:
!  NO = Normalized*Tadj*Padj*Fadj*Cadj
!
!  New:
!  NO   = f( T, biome, WFPS, Fert)  x Pulse(dryspell) x canopy uptake 
!  BDNSP Updates:
!  1 - Update moisture treatment: soil moisture as a continuous variable 
!  using WFPS rather than discrete wet/dry states and purely exponential 
!  T impact (impact = -1. Tg N/yr)
!
!  2 - Update to Fertilizer:  new fertilizer maps including chemical and 
!  manure fertilizer from Potter et al., [2010] distributed using MODIS EVI 
!  seasonality, online-N deposition as a fertilizer source, and N-fertilizer
!  source subject to T, WFPS, and pulsing like other N (impact = +1.3 Tg N/yr)
!
!  3- Update Pulsing Scheme: Yan et al., [2005] (shorter, stronger pulses) 
!  (impact = +1. Tg N/yr). Also added restart file containing dry spell 
!  information to properly account for dry spell length in continuing runs.
!
C    Information needed to estimate NO emissions:
C       Julian Day          (integer)    JDATE
C       Surface Temperature (MCIP field) TA    (K)
C       Rainfall    (MCIP derived field) RAIN  (cm)
C       Soil Moisture       (MCIP field) SOILM (M**3/M**3) (PX_VERSION)
C            (ratio of volume of water per volume of soil)
C       Soil Temperature    (MCIP field) SOILT (K)         (PX_VERSION)
C       Soil Type           (MCIP field) ISLTYP            (PX_VERSION)
C       Saturation values for soil types (constants)       (PX_VERSION)
C    FOR PX Version, the Temperature adjustment factor accounts for wet and dry
C    soils and the precipitation adjustment factor accounts for saturated soils
C    FOR the non-PX version, the basic algorithm remains with a temperature
C    adjustment factor (dry soil) and no adjustment for saturated soils
 
C    The following arrays are potentially updated after a call to HRNO:
C       PTYPE     type of NO emission pulse 
C       PULSEDATE julian date for the beginning of an NO pulse 
C       PULSETIME        time for the beginning of an NO pulse
   
C    The calculations are based on the following paper:
C    Hudman, R.C., N.E. Moore, Steps toward a mechanistic model
!    of global soil nitric oxide emissions: implementation and space
!    based-constraints, ACPD, Manuscript under review for ACP  
 
C    References:
!  \begin{itemize}
!  \item Wang, Y., D.J. Jacob, and J.A. Logan, Global simulation of 
!        tropospheric O3-NOx-hydrocarbon chemistry, 1. Model formulation, 
!        J. Geophys. Res., 103/D9, 10,713-10,726, 1998.
!  \item Yienger, J.J, and H. Levy, \emph{Empirical model of global 
!        soil-biogenic NOx emissions}, \underline{J. Geophys. Res.}, 
!        \textbf{100}, D6, 11,447-11464, June 20, 1995.
!  \item Yan, X., T. Ohara, and H. Akimoto,\emph{ Statistical modeling 
!        of global soil NOx emissions},\underline{Global Biogeochem. Cycles},
!          \textbf{19}, GB3019, doi:10.1029/2004GB002276, 2005.
!  \item Potter, P., Ramankutty, N., Bennett, E.,  and Donner, S.: 
!        Characterizing the Spatial Patterns of Global Fertilizer 
!        Application and Manure Production, Earth Interactions, 
!        in press, 2010.
!  \item Moore, N.E., Improving global bottom-up biogenic soil NOx 
!        inventories, Master's Thesis, Dalhousie University, 2007.
!  \item Hudman, R.C., N.E. Moore, Steps toward a mechanistic model
!        of global soil nitric oxide emissions: implementation and space
!        based-constraints, ACPD, Manuscript under review for ACP
!  \end{itemize}
!
! !INTERFACE: 
!188424.00 
C    

C Preconditions:
C     Normalized NO emissions, Surface Temperature, Soil Moisture, Soil type,
C     NO emission pulse type, soil moisture from previous time step, julian date
C     of NO emission pulse start, time of NO emission pulse start,
C     soil type, SOIL TYPES, Land use data
 
C Subroutines and Functions Called (directly or indirectly):
C     PRECIP_ADJ     computes precipitation adjustment factor
C     FERTILIZER_ADJ computes fertlizer adjustment factor
C     VEG_ADJ        computes vegetation adjustment factor
C     GROWSEASON     computes Julian day of growing season
C     PRECIPFAC      computes precip adjustment factor from rainfall in last 24 hrs
C                    and time since pulse initiation
C     PULSETYPE      determines type & duration of NO emission pulse from rainrate
      
C Revision History:
C    10/01: Prototype by GAP
C    10/03: modified transition to non growing season for jul-oct of the year
C    08/04: Converted to SMOKE code style by C Seppanen
C   Mar 07: Restructure; J.Young
C   Jan 26: J.Young - move input data reads from tmpbeis;
C                     remove ck & report if TAIR > 315;
C                     restructure growing season, col/row loops;
C                     restructure PRECIP_ADJ
C   Jan 27: D. Wong - Eliminate potential race condition with mype = 0 and
C                     barrier implementation - not needed anyway.
C    02/11: S.Roselle-Replaced I/O API include files with UTILIO_DEFN
C    05/11: D. Wong - incorporated twoway model implementation
C    05/12: J.Young - make PX_version the default
  
C-----------------------------------------------------------------------
C Modified from:

C Project Title: Sparse Matrix Operator Kernel Emissions (SMOKE) Modeling System
C File: @(#)$Id: hrnoBDSNP.F,v 1.1.1.1 2012/05/10 17:23:38 sjr Exp $
C COPYRIGHT (C) 2004, Environmental Modeling for Policy Development
C All Rights Reserved
C Carolina Environmental Program
C University of North Carolina at Chapel Hill
C 137 E. Franklin St., CB# 6116
C Chapel Hill, NC 27599-6116
C smoke@unc.edu
C Pathname: $Source: /home/qzr1/CMAQv5.0.2/models/CCTM/biog/beis3/hrno.F,v $
C Last updated: $Date: 2012/05/10 17:23:38 $ 
C-----------------------------------------------------------------------

      USE HGRD_DEFN             ! horizontal domain specifications
      USE BIOG_EMIS, ONLY: NSEF ! beis NSEF=number species, NO is last
      USE UTILIO_DEFN
      USE CANOPY_NOX_MOD
      IMPLICIT NONE
        
C Includes:

C Arguments:
      INTEGER, INTENT( IN )  :: JDATE             ! current simulation date (YYYYDDD)
      INTEGER, INTENT( IN )  :: JTIME             ! current simulation time (HHMMSS)
      INTEGER, INTENT( IN )  :: TSTEP( 3 )        ! time step vector (HHMMSS)
      INTEGER, INTENT( IN ) :: NC      ! no. columns
      INTEGER, INTENT( IN ) :: NR      ! no. rows
C     These are arrays 
      REAL,    INTENT( IN ) :: COSZEN( NC,NR )        ! cosine of zenith angle
      REAL,    INTENT( IN ) :: TASFC    ( NC,NR )        ! surface air temperature [K]
      REAL,    INTENT( IN ) :: SSOLAR( NC,NR )        ! surface radiation [w/m**2]
      REAL,    INTENT( IN ) :: PRES  ( NC,NR )        ! surface pressure [Pa]
      REAL,    INTENT( OUT ) :: EMPOL( NCOLS,NROWS,NSEF ) ! output NO emissions gramsN/hour

C External Functions
      LOGICAL,         EXTERNAL :: CHKGRID

C Parameters:
C      INTEGER, PARAMETER :: MXRHRS = 24     ! no. of rainfall hours for YL95 algorithm
      INTEGER, PARAMETER :: MAXSTYPES = 11
!                 IF ( SOILCAT .LE. 14 ) THEN  <<< NEW PX METCRO2D ??? 14 = water
C      REAL,    PARAMETER :: CFNODRYFC = ( 1.0 / 3.0 ) * ( 1.0 / 30.0 )
      
      ! Scale factor so that fertilizer emission = 1.8 Tg N/yr (Stehfest and Bouwman, 2006)
      ! before canopy reduction
      REAL*8, PARAMETER :: FERT_SCALE = 0.0068 
      ! Value calculated by running the 2x2.5 GEOS-Chem model
      
      REAL*8,  PARAMETER :: TAU_MONTHS   = 6. ! this is the decay time for dep. N reservoir, fert is 4 months
      REAL*8,  PARAMETER :: SECPERDAY    = 86400.d0
      REAL*8,  PARAMETER :: DAYSPERMONTH = 30.
      REAL*8,  PARAMETER :: TAU_SEC      = TAU_MONTHS * DAYSPERMONTH * SECPERDAY
      
      ! Conversion factor from [ng N/m2/s] to [molec/cm2/s]
      REAL*8,  PARAMETER :: UNITCONV = 4.3d9
      
      REAL*8,  PARAMETER :: CELLAREA = 12000.0*12000.0 !12km x 12km grid in meters
                                       !XCELL_GD * YCELL_GD compile error, testing

      ! New soil biomes based on Steinkamp et al., 2011
      INTEGER, PARAMETER :: NSOIL    = 24

      ! Canopy wind extinction coefficients
      ! (cf. Yienger & Levy [1995], Sec 5), now a function of the MODIS/KOPPEN biometype (J.D. Maasakkers)
       REAL*8,  PARAMETER :: SOILEXC(NSOIL)    = (/ 
     &  0.10, 0.50, 0.10, 0.10, 0.10,
     &  0.10, 0.10, 0.10, 0.10, 1.00,
     &  1.00, 1.00, 1.00, 2.00, 4.00,
     &  4.00, 4.00, 4.00, 4.00, 4.00,
     &  4.00, 2.00, 0.10, 2.00                  /)

      ! Steinkamp and Lawrence, 2011 A values, wet biome coefficients
      ! for each of the 24 soil biomes [ng N/m2/s] 
      REAL*8,  PARAMETER :: A_BIOME(NSOIL)          =                (/ 
     &   0.00, 0.00, 0.00, 0.00, 0.00, 0.06, 0.09, 0.09, 0.01, 
     &   0.84, 0.84, 0.24, 0.42, 0.62, 0.03, 0.36, 0.36, 0.35, 
     &   1.66, 0.08, 0.44, 0.57, 0.57, 0.57     /)
        
C Saturation values for 11 soil types from pxpbl.F  (MCIP PX version)
C Pleim-Xiu Land-Surface and PBL Model (PX-LSM)
C See Jacquemin B. and Noilhan J. (1990), Bound.-Layer Meteorol., 52, 93-134.

      REAL :: SATURATION( MAXSTYPES ) =
     &                  (/ 0.395, 0.410, 0.435, 0.485,
     &                     0.451, 0.420, 0.477, 0.476,
     &                     0.426, 0.482, 0.482 /)

C Local Variables:

      CHARACTER( 16 ), SAVE :: MNAME   ! logical name for MET_CRO_2D
      CHARACTER( 16 ), SAVE :: NNAME   ! logical name for normalized-emissions input
      CHARACTER( 16 ), SAVE :: SOILINSTATE = 'SOILINSTATE' ! logical name for input NO soil data, restart file
      CHARACTER( 16 ), SAVE :: SOILOUT = 'SOILOUT' ! logical name for output NO soil data - same format as soilinstate
C Land use files for BDSNP: both time independant in CMAQ sense and absolutely - e.g. fertilizer does not vary with year
      CHARACTER( 16 ), SAVE :: CLIMAFILE = 'CLIMAFILE' ! climate, arid 
      CHARACTER( 16 ), SAVE :: CLIMNAFILE = 'CLIMNAFILE' ! climate, nonarid 
      CHARACTER( 16 ), SAVE :: LANDFRACFILE = 'LANDFRACFILE' ! biome type 
      CHARACTER( 16 ), SAVE :: FERTRESFILE = 'FERTRESFILE' ! fertilizer reservoir
      CHARACTER( 16 ), SAVE :: EPICRESFILE = 'EPICRESFILE' ! Epic file
      CHARACTER( 16 ), SAVE, ALLOCATABLE :: DDTTM( : ) ! description date and time
C      CHARACTER( 33 ), SAVE         !123456789012345678901234567890123
C     &                 :: DESCSTR = 'hrly cnv. & non-cnv. rainfall for'

      CHARACTER( 16 ) :: VAR        ! variable name

      INTEGER, SAVE :: IHR       ! current simulation hour

      REAL,    ALLOCATABLE, SAVE :: SOILM    ( :,: )  ! soil moisture [m3/m3] (PX)
      REAL,    ALLOCATABLE, SAVE :: SOILMPREV( :,: )  ! soil moisture previous tstep [m3/m3] (PX)
      REAL,    ALLOCATABLE, SAVE :: SOILT    ( :,: )  ! soil temperature [K] (PX)
      REAL,    ALLOCATABLE, SAVE :: ISLTYP   ( :,: )  ! soil type (PX)
      REAL,    ALLOCATABLE, SAVE :: FERT     ( :,: )  ! "ng N m-2" already - reservoir
      REAL,    ALLOCATABLE, SAVE :: T1_EPIC    ( :,: )  ! "Epic input ng N m-2"
      REAL,    ALLOCATABLE, SAVE :: EPICN    ( :,: )  !"epic N reservoir ng N m-2"
      REAL,    ALLOCATABLE, SAVE :: NDEPRES  ( :,: )  ! "ng N m-2" deposition reservoir
      REAL,    ALLOCATABLE, SAVE :: DRYPERIOD( :,: )  ! Will determine pulse, hours
      REAL,    ALLOCATABLE, SAVE :: PFACTOR  ( :,: )  ! current pulse factor
C Gridded Canopy NOx reduction factor for BDSNP Soil NO calculations
      REAL,    ALLOCATABLE, SAVE :: CRF   ( :,: )     ! 0-1

C --- diagnostic variables, can be removed in final version      
      REAL,    ALLOCATABLE, SAVE :: CRFAVG   ( :,: )  ! 0-1
      REAL,    ALLOCATABLE, SAVE :: PULSEAVG   ( :,: )  ! 1+
      REAL,    ALLOCATABLE, SAVE :: BASESUM   ( :,: )  ! used in calculating the above two averages
      REAL,    ALLOCATABLE, SAVE :: THETA_DIAG( :,: )  ! diagnositc theta
      REAL,    ALLOCATABLE, SAVE :: WET_DIAG ( :,: )  ! diagnositc wet term
      REAL,    ALLOCATABLE, SAVE :: TEMP_DIAG ( :,: )  ! diagnositc temp term
      REAL,    ALLOCATABLE, SAVE :: A_DIAG ( :,: )  ! diagnositc biome base emissions term
      REAL,    ALLOCATABLE, SAVE :: AFERT_DIAG ( :,: )  ! diagnositc fert emissions term
      REAL,    ALLOCATABLE, SAVE :: NRES_FERT_DIAG ( :,: )  ! diagnositc nres fert
      REAL,    ALLOCATABLE, SAVE :: NRES_DEP_DIAG ( :,: )  ! diagnositc nres  dep    
C ---------------------------------------------------------------------------      
      INTEGER, ALLOCATABLE, Save :: ARID     ( :,: )
      INTEGER, ALLOCATABLE, Save :: NONARID  ( :,: )
      INTEGER, ALLOCATABLE, Save :: LANDFRAC ( :,: )
      REAL,                 SAVE :: EMPOLSUM, EMPOLAVG       ! use to check reasonableness of results
      REAL,                 SAVE :: TIMECHECK ! use to output CPU_TIME(TIMECHECK) to see if this section of code is running unreasonably long
      
      INTEGER, SAVE :: COUNTER ! save number of subroutine calls for calculating average CRF or others

      
      INTEGER, SAVE :: EDATE     ! end scenario date
      INTEGER, SAVE :: ETIME     ! end scenario time
      INTEGER, SAVE :: NDATE     ! test date to update rainfall
      INTEGER, SAVE :: NTIME     ! test time to update rainfall
      INTEGER, SAVE :: SDATE     ! scenario start date
      INTEGER, SAVE :: STIME     ! scenario start time
        
      LOGICAL, SAVE :: INITIAL_RUN            ! false: use SOILINSTATE restart file
                                              ! initial_run is set in run.cctm script
C      LOGICAL, SAVE :: INITIAL_DAY = .FALSE.  ! true: 1st 24 hours; no previous data
                                              ! false: previous 24 hours of rainfall
                                              ! are available for HRNO

      LOGICAL, SAVE :: PX_VERSION         ! true: use PX version of MCIP; should always be true

      INTEGER          SOILCAT            ! soil category
      INTEGER          NSTEPS             ! run duration (HHMMSS)
      INTEGER, SAVE :: MSTEPS             ! run no. of steps
      INTEGER          I, J, K, R, C, L   ! counters
      INTEGER          JDAY, YEAR
      CHARACTER*3      CHARDAY
      LOGICAL          OK
      INTEGER          IOS                ! IO or memory allocation status
      
C      REAL,    SAVE :: EFAC
       REAL            TEMP_TERM
       REAL            WET_TERM
       REAL            PULSE
       REAL            A_FERT
       REAL            CRF_TERM
       REAL            SOILNOX, FERTDIAG
C      REAL             CFNO               ! NO correction factor
C      REAL             CFNOGRASS          ! NO correction factor for grasslands
C      REAL             TAIR               ! surface temperature
C      REAL             TSOI               ! soil temperature
       REAL             THETA              ! water filled pore space
       REAL             THETAPREV

C      REAL             CFNOWET, CFNODRY, THETA
C      REAL             FAC1, FAC2, FAC3, FAC4

      LOGICAL, SAVE :: USE_SOILT = .TRUE. ! use soil temperature in PX version
                                          ! rather than estimate as in BEIS2

      INTEGER, SAVE :: LOGDEV
      LOGICAL, SAVE :: FIRSTIME = .TRUE.
      CHARACTER( 256 ) :: MESG            ! message buffer
      CHARACTER( 16 )  :: PNAME = 'BDSNPHRNO'  ! procedure name

      INTEGER      GXOFF, GYOFF           ! global origin offset from file
C for INTERPX
      INTEGER       :: STRTCOLNB3, ENDCOLNB3, STRTROWNB3, ENDROWNB3  ! Normalized BEIS3
      INTEGER       :: STRTCOLSIN, ENDCOLSIN, STRTROWSIN, ENDROWSIN  ! SOILINSTATE
      INTEGER, SAVE :: STRTCOLMC2, ENDCOLMC2, STRTROWMC2, ENDROWMC2  ! MET_CRO_2D
      INTEGER, SAVE :: STRTCOLBDS, ENDCOLBDS, STRTROWBDS, ENDROWBDS  ! BDSNP files
      INTEGER, SAVE :: STRTCOLEPI, ENDCOLEPI, STRTROWEPI, ENDROWEPI  !EPIC files




#ifdef Verbose
      integer mxptype, ncfno0
      real    mxrntot, mxfac1, avgfac2, mxfac3, mxfac4
      real    mxtair, mxcfno
#endif
C-----------------------------------------------------------------------
      
      IF ( FIRSTIME ) THEN
         FIRSTIME = .FALSE.
         BDSNP = .TRUE.
         ! we need to initialize and allocate:
         ! pulse
         ! length of dry period
         ! soil moisture of previous time step
         ! N reservoir, deposition only
         ! These values can be provided from a restart file. The restart file is 'timeless'.
         ! This means CMAQ isn't checking to see if the restart file is actually from 
         ! the immediately prior timstep.

         LOGDEV = INIT3()
         WRITE( LOGDEV,*) 'FIRSTIME BDSNP and LOGDEV=INIT3()',NCOLS,NROWS
C Determine last timestamp
         SDATE = ENVINT( 'CTM_STDATE', 'Scenario Start (YYYYJJJ)', 0, IOS )
         STIME = ENVINT( 'CTM_STTIME', 'Scenario Start (HHMMSS)', 0, IOS )
         NSTEPS = ENVINT( 'CTM_RUNLEN', 'Scenario Duration (HHMMSS)', 0, IOS )
         EDATE = SDATE; ETIME = STIME
         CALL NEXTIME( EDATE, ETIME, NSTEPS )   ! end date & time
         MSTEPS = TIME2SEC( NSTEPS ) / TIME2SEC( TSTEP( 1 ) )

C Check if using PX version of MCIP
         PX_VERSION = ENVYN( 'PX_VERSION', 'MCIP is PX version?',
     &                       .TRUE., IOS )
     
C     ! make sure it only runs with PX version
         IF( .NOT. PX_VERSION ) THEN
            MESG = "BDSNP Soil NO is only compatible with PX version"
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
         END IF
         
C Open met file
         MNAME = PROMPTMFILE(
     &           'Enter name for gridded met input file',
     &           FSREAD3, 'MET_CRO_2D', PNAME )
         Write(LOGDEV,*) 'opened met file', NCOLS, NROWS
C Get description of met file
         IF ( .NOT. DESC3( MNAME ) ) THEN
            MESG = 'Could not get description of file "'
     &           // TRIM( MNAME ) // '"'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
         END IF
         Write(LOGDEV,*) 'got desc met file', NCOLS, NROWS
C Check that grid description matches BGRD file
         IF ( .NOT. CHKGRID( MNAME ) ) THEN
            MESG = 'Grid in file "' // TRIM( MNAME )
     &           // '" does not match previously set grid.'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
         END IF
         Write(LOGDEV,*) 'checked desc met file', NCOLS, NROWS


C Get domain decomp info for the met file
         CALL SUBHFILE ( MNAME, GXOFF, GYOFF,
     &                   STRTCOLMC2, ENDCOLMC2, STRTROWMC2, ENDROWMC2 )
         Write(LOGDEV,*) 'domain decomp met file', NCOLS, NROWS
C Get normalized emissions file, BGRD
         NNAME = PROMPTMFILE(
     &           'Enter name for Normalized Emissions input file',
     &           FSREAD3, 'B3GRD', PNAME )
         Write(LOGDEV,*) 'prompt b3grd file', NCOLS, NROWS
C Read description of normalized emissions file
         IF ( .NOT. DESC3( NNAME ) ) THEN
            MESG = 'Could not get description of file "' //
     &             TRIM( NNAME ) // '"'
            CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
         END IF
         Write(LOGDEV,*) 'got desc b3grd file', NCOLS, NROWS
C Initialize grid definition
         OK = CHKGRID( NNAME )
         Write(LOGDEV,*) 'initialize grid def', NCOLS, NROWS         
C Get domain decomp info for the normalized emissions file
         CALL SUBHFILE ( NNAME, GXOFF, GYOFF,
     &                   STRTCOLNB3, ENDCOLNB3, STRTROWNB3, ENDROWNB3 )
         Write(LOGDEV,*) 'domain decomp b3grd file', NCOLS, NROWS
C Allocate memory for data and read
         Write(LOGDEV,*) 'about to allocate memory, NCOLS,NROWS', NCOLS, NROWS

         ALLOCATE( SOILM( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'SOILM', PNAME )
         
         ALLOCATE( SOILMPREV( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'SOILMPREV', PNAME )

         ALLOCATE( SOILT( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'SOILT', PNAME )

         ALLOCATE( ISLTYP( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'ISLTYP', PNAME )

         ALLOCATE( DRYPERIOD( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'PTYPE', PNAME )
         
         ALLOCATE( NDEPRES( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'PTYPE', PNAME )
         
         ALLOCATE( NDEPRATE( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'PTYPE', PNAME )
         
         NDEPRATE = 0.0

         ALLOCATE( PFACTOR( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'PFACTOR', PNAME )
         
         ALLOCATE( ARID( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'ARID', PNAME )
         
         ALLOCATE( NONARID( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'NONARID', PNAME )
         
         ALLOCATE( LANDFRAC( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'LANDFRAC', PNAME )
         
         ALLOCATE( FERT( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'FERT', PNAME )
                  
         ALLOCATE( T1_EPIC( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'T1_EPIC', PNAME )

         ALLOCATE( EPICN( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'EPICN', PNAME )

         ALLOCATE( CRF( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'CRF', PNAME )
         
         ALLOCATE( CRFAVG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'CRF', PNAME )
         
         ALLOCATE( PULSEAVG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'PULSEAVG', PNAME )
         
         ALLOCATE( BASESUM( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'BASESUM', PNAME )
         
C ------ Diagnostics -----------------------------------         
         ALLOCATE( THETA_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'THETA_DIAG', PNAME )
         
         ALLOCATE( WET_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'WET_DIAG', PNAME )
         
         ALLOCATE( TEMP_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'TEMP_DIAG', PNAME )
         
         ALLOCATE( A_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'A_DIAG(', PNAME )
         
         ALLOCATE( AFERT_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'AFERT_DIAG', PNAME )
         
         ALLOCATE( NRES_FERT_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'NRES_FERT_DIAG', PNAME )
         
         ALLOCATE( NRES_DEP_DIAG( NCOLS,NROWS ), STAT=IOS )
         CALL CHECKMEM( IOS, 'NRES_DEP_DIAG', PNAME )
         
C-----------------------------------------------------------------------------         
C Initial run if the model hasn't been run before, otherwise use a restart file
C to determine DRYPERIOD, pulse state, prev. timestep soil moisture, and N reservoir. 
         WRITE( LOGDEV,'(/5X, A)' ) 'Temporal BEIS ...'
         MESG = 'Initial run?'
         INITIAL_RUN = ENVYN( 'INITIAL_RUN', MESG, .FALSE., IOS )

C If initial run, initialize some variables, otherwise get them from file
         IF ( INITIAL_RUN ) THEN

            PFACTOR   = 1d0   ! array
            DRYPERIOD = 0d0   ! array
            
            SOILMPREV = 0d0   ! array
            FERT      = 0d0   ! array
            NDEPRES   = 0d0   ! array
            EMPOLSUM  = 0d0
            EMPOLAVG  = 0d0
            BASESUM = 0.0
            CRFAVG = 0.0
            PULSEAVG = 0.0
            !NDEPRES   = 0d0   ! array
            !attempt to use steady state condition to reduce spin up time by setting dN/dt = 0
            !or NDEPRES = Dep rate * tau, the decay time
            DO R = 1, NROWS
                DO C = 1, NCOLS
                NDEPRES( C,R ) = NDEPRATE( C,R )*TAU_SEC
C           check for negatives
          IF( NDEPRES(C,R) .lt. 0.0 ) THEN
          Write(LOGDEV,*) 'NDEPRES negative', NDEPRES, ' ', NDEPRATE(C,R)
          MESG = 'negative'
          CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
          ELSE IF( NDEPRATE(C,R) .lt. 0.0 ) THEN
          Write(LOGDEV,*) 'NDEPRATE negative', NDEPRES, ' ',NDEPRATE(C,R)
          CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
          END IF
                END DO
            END DO
         ELSE ! SOILINSTATE file available

C Open saved NO soil state data file
            SOILINSTATE = PROMPTMFILE(
     &                'Enter name for NO EMISSIONS SAVE file',
     &                FSREAD3, 'SOILINSTATE', PNAME )

C Get description of NO soil state data file
            IF ( .NOT. DESC3( SOILINSTATE ) ) THEN
               MESG = 'Could not get description of file "' //
     &                 TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
            END IF

C Check that the file start date and time are consistent
            IF ( SDATE3D .NE. SDATE ) THEN
               WRITE( MESG,94010 ) 'Cannot use SOILINSTATE file; ' //
     &             'found date ', SDATE3D, ' expecting ', SDATE
               CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
            END IF

            IF ( STIME3D .NE. STIME ) THEN
               WRITE( MESG,94010 ) 'Cannot use SOILINSTATE file; ' //
     &             'found time ', STIME3D, ' expecting ', STIME
               CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
            END IF

C Get domain decomp info for the NO soil state data file
            CALL SUBHFILE ( SOILINSTATE, GXOFF, GYOFF,
     &                      STRTCOLSIN, ENDCOLSIN, STRTROWSIN, ENDROWSIN )

C Read data from file
            VAR = 'DRYPERIOD'
            IF ( .NOT. XTRACT3( SOILINSTATE, VAR,
     &                          1,1, STRTROWSIN,ENDROWSIN, STRTCOLSIN,ENDCOLSIN,
     &                          0, 0, DRYPERIOD ) ) THEN
               MESG = 'Could not read "' // TRIM( VAR ) //
     &                '" from file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF

            VAR = 'PFACTOR'
            IF ( .NOT. XTRACT3( SOILINSTATE, VAR,
     &                          1,1, STRTROWSIN,ENDROWSIN, STRTCOLSIN,ENDCOLSIN,
     &                          0, 0, PFACTOR ) ) THEN
               MESG = 'Could not read "' // TRIM( VAR ) //
     &                '" from file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF

            VAR = 'SOILMPREV'
            IF ( .NOT. XTRACT3( SOILINSTATE, VAR,
     &                          1,1, STRTROWSIN,ENDROWSIN, STRTCOLSIN,ENDCOLSIN,
     &                          0, 0, SOILMPREV ) ) THEN
               MESG = 'Could not read "' // TRIM( VAR ) //
     &                '" from file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF

            VAR = 'NDEPRES'
            IF ( .NOT. XTRACT3( SOILINSTATE, VAR,
     &                          1,1, STRTROWSIN,ENDROWSIN, STRTCOLSIN,ENDCOLSIN,
     &                          0, 0, NDEPRES ) ) THEN
               MESG = 'Could not read "' // TRIM( VAR ) //
     &                '" from file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF
            
             VAR = 'NDEPRATE_DIAG'
            IF ( .NOT. XTRACT3( SOILINSTATE, VAR,
     &                          1,1, STRTROWSIN,ENDROWSIN, STRTCOLSIN,ENDCOLSIN,
     &                          0, 0, NDEPRATE ) ) THEN
               MESG = 'Could not read "' // TRIM( VAR ) //
     &                '" from file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF           
            
C Close input file
            IF ( .NOT. CLOSE3( SOILINSTATE ) ) THEN
               MESG = 'Could not close file "' // TRIM( SOILINSTATE ) // '"'
               CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
            END IF

         END IF   ! initial run check
C         Write(LOGDEV,*) 'about read bdsnp data, NCOLS,NROWS', NCOLS, NROWS
         
         
C Get domain decomp info for the BDSNP files
         CALL SUBHFILE ( CLIMAFILE, GXOFF, GYOFF,
     &                   STRTCOLBDS, ENDCOLBDS, STRTROWBDS, ENDROWBDS )
     
C Read Time Independant BDSNP specific data  
      Write(LOGDEV,*) 'about to read bdsnp data:', GXOFF, GYOFF,
     &                   STRTCOLBDS, ENDCOLBDS, STRTROWBDS, ENDROWBDS
       
      IF ( .NOT. OPEN3( CLIMAFILE, FSREAD3, PNAME ) ) 
     & THEN
         MESG = 'Could not open ClimA file'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
      END IF   

      IF ( .NOT. XTRACT3( CLIMAFILE, 'Arid',
     & 1,1, STRTROWBDS,ENDROWBDS, STRTCOLBDS,ENDCOLBDS,
     & 0, 0, ARID ) ) THEN
       MESG = 'Could not read Arid from ClimA'
       CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 ) 
      END IF  
      
      IF ( .NOT. OPEN3( CLIMNAFILE, FSREAD3, PNAME ) ) 
     & THEN
         MESG = 'Could not open ClimNA file'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
      END IF
      
      IF ( .NOT. XTRACT3( CLIMNAFILE, 'NonArid',
     & 1,1, STRTROWBDS,ENDROWBDS, STRTCOLBDS,ENDCOLBDS,
     & 0, 0, NONARID ) ) THEN
       MESG = 'Could not read NonArid from ClimNA'
       CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
      END IF
      
      IF ( .NOT. OPEN3( LANDFRACFILE, FSREAD3, PNAME ) ) 
     & THEN
         MESG = 'Could not open LANDFRAC file'
         CALL M3EXIT( PNAME,JDATE, JTIME, MESG, XSTAT1 )
      END IF
C      Write(LOGDEV,*) 'about to extract lfrac, NCOLS,NROWS', NCOLS, NROWS
      IF ( .NOT. XTRACT3( LANDFRACFILE, 'LANDFRAC',
     & 1,1, STRTROWBDS,ENDROWBDS, STRTCOLBDS,ENDCOLBDS,
     & 0, 0, LANDFRAC ) ) THEN
       MESG = 'Could not read LANDFRAC from LANDFRACFILE'
       CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
      END IF
      
      WRITE( LOGDEV,*) 'finished IF FIRSTIME BDSNP'
      END IF   ! FIRSTIME
      
      IF ( SECSDIFF( JDATE,JTIME, EDATE,ETIME ) .LE. TIME2SEC( TSTEP( 2 ) ) ) GO TO 9999
      
C Soil moisture, temperature, type relies on PX version      

C Read soil moisture data
         IF ( .NOT. INTERPX( MNAME, 'SOIM1', PNAME,
     &                       STRTCOLMC2,ENDCOLMC2, STRTROWMC2,ENDROWMC2, 1,1,
     &                       JDATE, JTIME, SOILM ) ) THEN
            MESG = 'Could not read "' // 'SOIM1' //
     &             '" from file "' // TRIM( MNAME ) // '"'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
         END IF

C Read soil temperature data
         IF ( .NOT. INTERPX( MNAME, 'SOIT1', PNAME,
     &                       STRTCOLMC2,ENDCOLMC2, STRTROWMC2,ENDROWMC2, 1,1,
     &                       JDATE, JTIME, SOILT ) ) THEN
            MESG = 'Could not read "' // 'SOIT1' //
     &                '" from file "' // TRIM( MNAME ) // '"'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
         END IF

C Read soil type data
         IF ( .NOT. INTERPX( MNAME, 'SLTYP', PNAME,
     &                       STRTCOLMC2,ENDCOLMC2, STRTROWMC2,ENDROWMC2, 1,1,
     &                       JDATE, JTIME, ISLTYP ) ) THEN
            MESG = 'Could not read "' // 'SLTYP' //
     &             '" from file "' // TRIM( MNAME ) // '"'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
         END IF

C Choose EPIC or Potter for the fertilizer data
      MESG = 'Use EPIC outputs for fertilizer (+ deposition) data?'
      EPIC = ENVYN ( 'EPIC', MESG, .FALSE., IOS )  
C     WRITE(LOGDEV, *), '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'        
C      WRITE(LOGDEV, *), 'EPIC =', EPIC
C      WRITE(LOGDEV, *), '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'        

      IF ( EPIC ) THEN
C      WRITE(LOGDEV, *) '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
C      WRITE(LOGDEV, *) 'Reading fertilizer files from EPIC outputs...'

C Get domain decomp info for the EPIC fertilizer files
       CALL SUBHFILE ( EPICRESFILE, GXOFF, GYOFF,
     &                 STRTCOLEPI, ENDCOLEPI, STRTROWEPI, ENDROWEPI )

C Read Time EPIC specific data  
       Write(LOGDEV,*) 'about to read EPIC data:', GXOFF, GYOFF,
     &                   STRTCOLEPI, ENDCOLEPI, STRTROWEPI, ENDROWEPI

C     read EPIC fertilizer reservoir
        IF ( .NOT. OPEN3( EPICRESFILE, FSREAD3, PNAME ) ) THEN
          MESG = 'Could not open EPIC Res file'
          CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
        END IF
C get the day's fertilizer from EPIC

        IF ( .NOT. XTRACT3( EPICRESFILE, 'T1_EPIC',
     &       1,1, STRTROWEPI,ENDROWEPI, STRTCOLEPI,ENDCOLEPI,
     &       JDATE, 0, T1_EPIC ) ) THEN
          MESG = 'Could not read '// 'T1_EPIC' // ' from EPICRESFILE'
        CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
        END IF


        DO R = 1, NROWS
         DO C = 1, NCOLS
             EPICN( C,R ) =  T1_EPIC( C,R )

          END DO
        END DO

      
      ELSE ! EPIC logic variable is FALSE

C      WRITE(LOGDEV, *) '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
C      WRITE(LOGDEV, *) 'Reading fertilizer files from Potter et al...'

C     read day dependant fertilizer reservoir e.g. Potter et al 2010
        IF ( .NOT. OPEN3( FERTRESFILE, FSREAD3, PNAME ) ) THEN
           MESG = 'Could not open Fert file'
           CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
        END IF
C get the day for the file
        YEAR = INT( FLOAT( JDATE ) / 1000.0 )
        JDAY = JDATE - YEAR * 1000
        WRITE(CHARDAY,'(i3.3)') JDAY
        VAR =  'FERTDAY'// CHARDAY
C get the day's fertilizer
        IF ( .NOT. XTRACT3( FERTRESFILE, VAR,
     &       1,1, STRTROWBDS,ENDROWBDS, STRTCOLBDS,ENDCOLBDS,
     &       0,0, FERT ) ) THEN
             MESG = 'Could not read '// VAR// ' from FERTRESFILE'
           CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT1 )
        END IF

      ENDIF

C Fertilizer N reservoir already calculated and read from file, update deposition reservoir from dep rate
      DO R = 1, NROWS
         DO C = 1, NCOLS
            CALL GET_NDEPRES( TSTEP, NDEPRES( C,R ), TAU_SEC, C, R )
         END DO
      END DO
      WRITE( LOGDEV,*) 'READ ALL DATA BDSNP'
      
C Calculate temporal non-speciated soil NO emissions to EMPOL  

C attempt to remove artefact by setting inputs to constant value
C      LANDFRAC = 1
C     If False Don't do any calculations to test - replicate 0 output
      IF( .TRUE. ) THEN
      CALL GET_CANOPY_NOX(JDATE, JTIME, NC, NR, COSZEN,
     & TASFC, SSOLAR, PRES, LANDFRAC, CRF)
     
      DO R = 1, NROWS
         DO C = 1, NCOLS
            
            SOILNOX  = 0d0
            FERTDIAG = 0d0
            
      ! ------Loop Over MODIS/Koppen  LANDFRACs
C            DO K = 1, 24
            
C            IF ( LANDFRAC( C,R ) .EQ. K ) THEN !Skip LANDFRACs not present
            K = LANDFRAC( C,R ) !Skip LANDFRACs not present
            ! Temperature-dependent term of soil NOx emissions [unitless]
            ! Uses PX soil temperature instead of inferring from air temperature
            TEMP_TERM = SOILTEMP( SOILT(C,R) )
            
            ! Use THETA instead of boolean wet/dry climate
            SOILCAT = INT( ISLTYP( C,R ) )
            IF ( SOILCAT .LE. MAXSTYPES ) THEN !not water
               THETA = SOILM( C,R ) / SATURATION( SOILCAT )
C               WRITE( LOGDEV,*) 'about to fill THETAPREV: ', THETAPREV !this seems to be the problem
               THETAPREV = SOILMPREV( C,R ) / SATURATION( SOILCAT )
C               WRITE( LOGDEV,*) 'filled THETAPREV: ', THETAPREV !this seems to be the problem
               ! Soil moisture scaling of soil NOx emissions 
               WET_TERM = SOILWET( THETA , ARID( C,R ), NONARID( C,R ))
            ELSE
               WET_TERM = 0d0
               THETA = 0d0
            END IF
            
            ! Cumulative multiplication factor (over baseline emissions) 
            ! that accounts for soil pulsing
C            IF ( (theta .ne. 0d0) .and. (R .eq. 50)) THEN
            ! Cumulative multiplication factor (over baseline emissions) 
            ! that accounts for soil pulsing
C            WRITE( LOGDEV,*) 'About to calculate PFACTOR'
C            WRITE( LOGDEV,*) 'THETA: ', THETA
C            WRITE( LOGDEV,*) 'TIME: ', REAL(TIME2SEC(TSTEP(2)))
C            WRITE( LOGDEV,*) 'THETAPREV: ', THETAPREV !this seems to be the problem
C            WRITE( LOGDEV,*) 'PFACTOR: ', PFACTOR( C,R ) !ran with this)
C            WRITE( LOGDEV,*) 'DRYPERIOD: ', DRYPERIOD( C,R ) !and this
      
C            MESG = 'test complete'
C            CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT1 )
C            END IF            
C            PULSE = 1d0 !apparently pulsing causes it to crash
            
            PULSE = PULSING( THETA, TSTEP, THETAPREV, 
     &                       PFACTOR( C,R ), DRYPERIOD( C,R ) )
C            IF ( PULSE .ne. 1d0 ) THEN !for testing the benchmark
C               WRITE( LOGDEV,*) 'THETA: ', THETA
C               WRITE( LOGDEV,*) 'TIME: ', REAL(TIME2SEC(TSTEP(2)))
C               WRITE( LOGDEV,*) 'THETAPREV: ', THETAPREV !this seems to be the problem
C               WRITE( LOGDEV,*) 'PFACTOR: ', PFACTOR( C,R ) !ran with this)
C               WRITE( LOGDEV,*) 'DRYPERIOD: ', DRYPERIOD( C,R ) !and this) 
C               WRITE( LOGDEV,*) 'PULSE: ', PULSE
C               WRITE( LOGDEV,*) 'C: ', C
C               WRITE( LOGDEV,*) 'R: ', R
C               MESG = 'PULSE not one'
C               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
C            END IF
                  
C            WRITE( LOGDEV,*) 'PFACTOR Calculated ', PFACTOR( C,R )
C            WRITE( LOGDEV,*) 'about to fertadd, NDEPRES:  ', 
C     &      NDEPRES( C,R ), ' ', FERT( C,R ), ' ', C, ' ', R
            ! Fertilizer emission


      IF ( EPIC ) THEN
C        WRITE(LOGDEV, *) '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
C        WRITE(LOGDEV, *) 'Using fertilizer files from EPIC outputs...'
           A_FERT = EPICRESERVOIRN( EPICN( C,R ) ) !adds epic N reservoir (deposition considered)
C            WRITE( LOGDEV,*) 'FERTADD complete ',A_FERT
      ELSE
C        WRITE(LOGDEV, *) '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
C        WRITE(LOGDEV, *) 'Using fertilizer files from Potter et al...'
            A_FERT = FERTADD( FERT( C,R ) , NDEPRES( C,R ) ) !adds reservoirs returns emission rates 
      END IF

   	    ! Canopy reduction factor
            CRF_TERM  = CRF( C,R ) 
C                  CRF_TERM  = SOILCRF( K, LAI, 
C     &                        R_CANOPY(K),
C     & 			      WINDSQR, SUNCOS )
                        
         !  SOILNOX includes fertilizer
            SOILNOX   = ( A_BIOME(K) + A_FERT )  !don't forget to check parenthesis when uncommenting
     &             * ( TEMP_TERM * WET_TERM * PULSE ) 
     &             * ( 1.d0 - CRF_TERM  ) 

         !  FERTDIAG, only used for the fertilizer diagnostic, note includes DEP
         ! not actually used for anything at the moment, only diagnostics
            FERTDIAG  = ( A_FERT )
     &             * ( TEMP_TERM * WET_TERM * PULSE ) 
     &             * ( 1.d0 - CRF_TERM  ) 
     
C            END IF !LANDFRAC check

C            ENDDO
           
           !scale emissions
           
           !diagnostics
           EMPOL( C,R,NSEF ) = SOILNOX * CELLAREA * 3600.0 * 10.0**-9 ![ng N/m2/s] * m2 * s/hr * g/ng
           ! sum various quantities for daily averaging 
           EMPOLSUM = EMPOLSUM + EMPOL(C,R,NSEF)
           BASESUM(C,R) = BASESUM(C,R) + ( A_BIOME(K) + A_FERT )  !don'tforget check paren when uncommenting
     &             * ( TEMP_TERM * WET_TERM) 
     &             * CELLAREA * 3600.0 * 10.0**-9 ![ng N/m2/s] * m2 * s/hr * g/ng
           PULSEAVG(C,R) = PULSEAVG(C,R) + ( A_BIOME(K) + A_FERT )  !don'tforget check paren when uncommenting
     &             * ( TEMP_TERM * WET_TERM * PULSE ) 
     &             * CELLAREA * 3600.0 * 10.0**-9 ![ng N/m2/s] * m2 * s/hr * g/ng
           CRFAVG(C,R) = CRFAVG(C,R) + ( A_BIOME(K) + A_FERT )  !don'tforget check paren when uncommenting
     &             * ( TEMP_TERM * WET_TERM ) 
     &             * ( 1.d0 - CRF_TERM  )
     &             * CELLAREA * 3600.0 * 10.0**-9 ![ng N/m2/s] * m2 * s/hr * g/ng
           
C--------- MORE DIAGNOSTICS  ---------------------------------         
           A_DIAG( C,R ) = A_BIOME(K)
           AFERT_DIAG( C,R ) = A_FERT
           NRES_FERT_DIAG( C,R ) = FERT( C,R )
           NRES_DEP_DIAG( C,R )  = NDEPRES( C,R )
           WET_DIAG( C,R ) = WET_TERM
           THETA_DIAG( C,R ) = THETA
           TEMP_DIAG( C,R ) = TEMP_TERM
           
C -----------------------------------------------------            
         END DO  ! columns
      END DO  ! rows 
      ELSE ! add things until it dies 
      WRITE( LOGDEV,*) 'BDSNP testing terms'
         DO R = 1, NROWS
         DO C = 1, NCOLS
            K = LANDFRAC( C,R ) !Skip LANDFRACs not present
            ! Temperature-dependent term of soil NOx emissions [unitless]
            ! Uses PX soil temperature instead of inferring from air temperature
            TEMP_TERM = SOILTEMP( SOILT(C,R) )
            ! Use THETA instead of boolean wet/dry climate
            SOILCAT = INT( ISLTYP( C,R ) )
            IF ( SOILCAT .LE. MAXSTYPES ) THEN !not water
               THETA = SOILM( C,R ) / SATURATION( SOILCAT )
               THETAPREV = SOILMPREV( C,R ) / SATURATION( SOILCAT )
               ! Soil moisture scaling of soil NOx emissions 
               WET_TERM = SOILWET( THETA , ARID( C,R ), NONARID( C,R ))
            ELSE
               WET_TERM = 0d0
               THETA = 0d0
            END IF            
            ! Cumulative multiplication factor (over baseline emissions) 
            ! that accounts for soil pulsing
C            PFACTOR( C,R ) = PULSING( THETA, TSTEP, THETAPREV, 
C     &                       PFACTOR( C,R ), DRYPERIOD( C,R ) )
            PULSE = PULSING( THETA, TSTEP, THETAPREV, 
     &                       PFACTOR( C,R ), DRYPERIOD( C,R ) )
     
         EMPOL( C,R,NSEF ) = 0.0
         END DO
         END DO
      END IF !  end do nothing test if
      
      SOILMPREV = SOILM !save soilM array to soilMprev for next time step
      WRITE( LOGDEV,*) 'BDSNP calculated emissions'
      CALL CPU_TIME(TIMECHECK)
      WRITE( LOGDEV,*) 'PROCESS TOOK:', TIMECHECK, 'SECONDS'
      EMPOLAVG = EMPOLSUM/FLOAT(NCOLS*NROWS)
      WRITE( LOGDEV,*) 'average value:', EMPOLAVG
      EMPOLSUM = 0d0 !array
      COUNTER = COUNTER + 1
      RETURN
      
9999  CONTINUE

C Create soil NO state save file at the end of the run for restart purposes

C Final timestamp
      NDATE = EDATE; NTIME = ETIME

C Build description for, and create/open soil NO emissions output file
      FTYPE3D = GRDDED3
      SDATE3D = NDATE
      STIME3D = NTIME
      TSTEP3D = 0   ! make it a time-independent file
      NCOLS3D = GL_NCOLS
      NROWS3D = GL_NROWS
      NLAYS3D = 1
      NVARS3D = 14 
      MXREC3D = 1

      VNAME3D = ' '
      VNAME3D( 1 ) = 'PFACTOR'
      VNAME3D( 2 ) = 'DRYPERIOD'
      VNAME3D( 3 ) = 'NDEPRES'
      VNAME3D( 4 ) = 'SOILMPREV'
C --- DIAGNOSTICS ---------------------------      
      VNAME3D( 5 ) = 'THETA_DIAG'
      VNAME3D( 6 ) = 'WET_TERM_DIAG'
      VNAME3D( 7 ) = 'TEMP_DIAG'
      VNAME3D( 8 ) = 'TEMP_TERM_DIAG'
      VNAME3D( 9 ) = 'A_DIAG'
      VNAME3D( 10 ) = 'NRES_FERT_DIAG'
      VNAME3D( 11 ) = 'AFERT_DIAG'
      VNAME3D( 12 ) = 'NDEPRATE_DIAG'
      VNAME3D( 13 ) = 'CRFAVG'
      VNAME3D( 14 ) = 'PULSEAVG'
c -------------------------------------------
      UNITS3D = ' '
      UNITS3D( 1 ) = 'REAL'
      UNITS3D( 2 ) = 'REAL'
      UNITS3D( 3 ) = 'REAL'
      UNITS3D( 4 ) = 'REAL'
      UNITS3D( 5 ) = 'REAL'
      UNITS3D( 6 ) = 'REAL'
      UNITS3D( 7 ) = 'REAL'
      UNITS3D( 8 ) = 'REAL'
      UNITS3D( 9 ) = 'REAL'
      UNITS3D( 10 ) = 'REAL'
      UNITS3D( 11 ) = 'REAL'
      UNITS3D( 12 ) = 'REAL'
      UNITS3D( 13 ) = 'REAL'
      UNITS3D( 14 ) = 'REAL'
 

      VDESC3D( 1 ) = 'NO emission current pulse factor'
      VDESC3D( 2 ) = 'length of the dry period in hours'
      VDESC3D( 3 ) = 'soil N reservoir from deposition'
      VDESC3D( 4 ) = 'Soil moisture prev. timestep m3/m3'
      VDESC3D( 5 ) = 'moisture WFPS 0-1'
      VDESC3D( 6 ) = 'moisture scale factore diagnostic'
      VDESC3D( 7 ) = 'temperature diagnostic'
      VDESC3D( 8 ) = 'temperature scale factor diagnostic'
      VDESC3D( 9 ) = 'biome base emission diagnostic'
      VDESC3D( 10 ) = 'NRES fert only diagnostic'
      VDESC3D( 11 ) = 'fertilizer emission factor diagnostic'
      VDESC3D( 12 ) = 'N deposition rate diagnostic'
      VDESC3D( 13 ) = 'canopy reduction factor diagnostic'
      VDESC3D( 14 ) = 'pulse factor diagnostic'

      VTYPE3D = 0
      VTYPE3D( 1:14 ) = M3REAL


      FDESC3D = ' '
      FDESC3D( 1 ) = 'Gridded soil state data for soil NO emissions'
      FDESC3D( 2 ) = '/From/ ' // PNAME
      FDESC3D( 3 ) = '/Version/ CMAQ'

C Open NO soil state save file
      IF ( MYPE .EQ. 0 ) THEN
         IF ( .NOT. OPEN3( SOILOUT, FSNEW3, PNAME ) ) THEN
            MESG = 'Could not open "' // TRIM( SOILOUT ) // '" file'
            CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )
         END IF
      END IF

C Write soil state file

      VAR = 'PFACTOR'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, PFACTOR ) ) THEN
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF

      VAR = 'DRYPERIOD'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, DRYPERIOD ) ) THEN
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF

      VAR = 'NDEPRES'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, NDEPRES ) ) THEN
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF

      VAR = 'SOILMPREV'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, SOILM ) ) THEN ! the end timestep's soilm becomes 
      !next time's soilmprev
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
C ---- DIAGNOSTICS -------------------------------------------------------------------     
      VAR = 'THETA_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, THETA_DIAG ) ) THEN ! diagnostic theta 
 
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'WET_TERM_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, WET_DIAG ) ) THEN ! diagnostic wet term 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'TEMP_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, SOILT ) ) THEN ! diagnostic temp 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'TEMP_TERM_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, TEMP_DIAG ) ) THEN ! diagnostic temp term 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'A_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, A_DIAG ) ) THEN ! diagnostic base emis
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'AFERT_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, AFERT_DIAG ) ) THEN ! diagnostic nres driven emis 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'NRES_FERT_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, NRES_FERT_DIAG ) ) THEN ! diagnostic 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'NDEPRATE_DIAG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, NDEPRATE ) ) THEN ! diagnostic 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'CRFAVG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, CRFAVG/BASESUM ) ) THEN ! diagnostic 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      VAR = 'PULSEAVG'
      IF ( .NOT. WRITE3( SOILOUT, VAR, NDATE, NTIME, PULSEAVG/BASESUM ) ) THEN ! diagnostic 
      
         MESG = 'Could not write "' // TRIM( VAR ) //
     &          '" to file "' // TRIM( SOILOUT ) // '"'
         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
      END IF
      
      COUNTER = 0
C ------------------------------------------------------------------------------------      
            WRITE( LOGDEV,94040 )
     &      'Timestep written to', SOILOUT,
     &      'for date and time', NDATE, NTIME
 

      RETURN

94010 FORMAT( A, F10.2, 1X, A, I3, ',', I3 )
94020 FORMAT( A, F10.2, 1X, A, I3, ',', I3, A )
94040 FORMAT( /5X, 3( A, :, 1X ), I8, ":", I6.6 )

      END SUBROUTINE HRNOBDSNP
C--------------------------------------------------------------------


      
         REAL FUNCTION PULSING( THETA, TSTEP, THETAPREV, 
     &                       PFACTOR, DRYPERIOD )!______
! !DESCRIPTION: Function PULSING calculates the increase (or "pulse") of 
!  soil NOx emission that happens after preciptiation falls on dry soil.  
!\\
!\\
!  According to  Yan et al., [2005] , this pulsing process is thought to  
!  be due to a release of inorganic nitrogen trapped on top of the dry soil 
!  and a subsequent reactivation of water-stressed bacteria, which then 
!  metabolize the excess nitrogen. This can happen in seasonally dry
!  grasslands and savannahs or over freshly fertilized fields.      
!  Soil NOx emissions consist of baseline emissions plus discrete "pulsing"
!  episodes.  We follow the Yan et al., [2005] algorithm, where the pulse
!  (relative to the flux pre wetting) is determined by the antecedent dry 
!  period, with a simple logarithmic relationship,
!
!  PFACTOR = 13.01 ln ( DRYPERIOD ) -  53.6
!
!  ,where PFACTOR is the magnitude of peak flux relative to prewetting flux, 
!  and DRYPERIOD  is the length of the antecedent dry period in hours.
! 
!  The pulse decays with 
!
!  PFACTOR = PFACTOR * EXP( -0.068d0 * TSTEP(HOURS) )      
      
         USE UTILIO_DEFN

         IMPLICIT NONE      
         
C Function arguments:
         INTEGER, INTENT( IN )    :: TSTEP( 3 )        ! time step vector (HHMMSS)
         REAL,    INTENT( IN )    :: THETA, THETAPREV  ! only avilable if PX version
         REAL,    INTENT( INOUT ) :: DRYPERIOD
         REAL,    INTENT( INOUT ) :: PFACTOR
C Local Variables
         REAL MOISTDIFF
         REAL DTHOURS
         DTHOURS = TIME2SEC(TSTEP(2))/3600.0
         ! If soil moisture less than 0.3 and no pulse is taking place
         IF ( THETA < 0.3D0 .and. PFACTOR == 1.D0) THEN

            ! Get change in soil moisture since previous timestep
            MOISTDIFF = ( THETA - THETAPREV )

            ! If change in soil moisture is > 0.01 (rains)
            IF ( MOISTDIFF > 0.01 ) THEN

               !Initialize new pulse factor (dry period hours)
               PFACTOR = 13.01 * LOG( DRYPERIOD ) - 53.6

               ! If dry period < ~3 days then no pulse
               IF ( PFACTOR < 1.0 ) PFACTOR = 1.0

                  ! Reinitialize dry period
                  DRYPERIOD = 0

                ! If no rain (i.e.,  change in soil moisture is < 0.01)
               ELSE
                ! Add one timestep to dry period
                DRYPERIOD = DRYPERIOD + DTHOURS

            ENDIF

         ! If box is already pulsing , then decay pulse one timestep
         ELSEIF ( PFACTOR /= 1.d0) THEN

            ! Decay pulse
            PFACTOR   = PFACTOR * EXP( -0.068d0 * DTHOURS )

            ! Update dry period
            IF ( THETA < 0.3D0 ) DRYPERIOD = DRYPERIOD + DTHOURS

            ! If end of pulse
            IF ( PFACTOR < 1.d0 ) PFACTOR = 1.d0
      
         ENDIF
         PULSING = PFACTOR
         RETURN
      
         END FUNCTION PULSING!_____
         
C -----------------------------------------------------------------------------------------------------------------      
   
         REAL FUNCTION SOILWET( THETA , ARID, NONARID)
C Calculate the soil moisture factor          
         USE UTILIO_DEFN
         
         IMPLICIT NONE
C Function arguments:
         REAL, INTENT( IN )       :: THETA !0-1 soil moisture 
         INTEGER, INTENT( IN )    :: ARID !1 indicates arid cell
         INTEGER, INTENT( IN )    :: NONARID !1 indicates nonarid cell, if both 0 then water        
C Local Variables

         IF ( ARID .EQ. 1 ) THEN !ARID, Max poison at theta = .2 
         SOILWET = 8.24*THETA*EXP(-12.5*THETA*THETA)
         ELSE IF (NONARID .EQ. 1 ) THEN !NONARID Max Poisson at theta = .3
         SOILWET = 5.5*THETA*EXP(-5.55*THETA*THETA)
         ELSE !neither arid nor nonarid, water or non-emitting cell
         SOILWET = 0.0
         END IF
         
         RETURN
         
         END FUNCTION SOILWET
C ---------------------------------------------------------------------------------------------------------         
         REAL FUNCTION SOILTEMP( SOILT )
C Calculate the soil temperature factor          
         USE UTILIO_DEFN
         
         IMPLICIT NONE
C Function arguments:
         REAL, INTENT( IN )       :: SOILT !kelvin, soil temperature
     
C Local Variables
         REAl SOILTC !temperature in degrees celsius 
         CHARACTER( 256 ) :: MESG            ! message buffer
         CHARACTER( 16 )  :: PNAME = 'SOILTEMP'  ! procedure name
         SOILTC = SOILT - 273.16

         IF ( SOILTC <= 0d0 ) THEN

         ! No soil emissions if temp below freezing
         SOILTEMP = 0d0
C         BENCHMARKING:         
C         MESG = 'temperature less than 0 in august florida?'
C         CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT1 )

         ELSE 

         ! Caps temperature response at 30C
         IF ( SOILTC >= 30.d0 ) SOILTC = 30.d0 
      
         SOILTEMP =  EXP( 0.103 * SOILTC )

         ENDIF
         
         RETURN
         
         END FUNCTION SOILTEMP
C ---------------------------------------------------------------------------------------------------------         
         REAL FUNCTION FERTADD( FERT , DEPN )
C Add fertilizer reservoir to deposition reservoir and create N driven emission factor        
         USE UTILIO_DEFN
         
         IMPLICIT NONE
C Function arguments:
         REAL, INTENT( IN )       :: FERT !fertilizer reservoir [ng N/m2]
         REAL, INTENT( IN )       :: DEPN !deposition reservoir [ng N/m2]
     
C Local Variables
         REAL*8,  PARAMETER :: SECPERYEAR    = 86400.d0 * 365.
         ! Scale factor so that fertilizer emission = 1.8 Tg N/yr (Stehfest and Bouwman, 2006)
         ! before canopy reduction
         REAL*8, PARAMETER :: FERT_SCALE = 0.0068 
         ! Value calculated by running the 2x2.5 GEOS-Chem model
         ! (J.D. Maasakkers)
         FERTADD = FERT + DEPN
         FERTADD = FERTADD / SECPERYEAR * FERT_SCALE

         
         RETURN
         
         END FUNCTION FERTADD         
C -----------------------------------------------------------------------------------------------------------------      
         REAL FUNCTION EPICRESERVOIRN( EPICN )
C Add EPIC N reservoir (FERT+DEP) in lieu to FERTADD taking from Potter
C data
         USE UTILIO_DEFN

         IMPLICIT NONE
C Functions arguments:
         REAL, INTENT( IN )       :: EPICN !ConvertedTo'ngN/m2'from'kg/ha
C         REAL, INTENT( IN )       :: DEPN !deposition reservoir[ngN/m2]

C Local Variables
         REAL*8,  PARAMETER :: SECPERYEAR_EPIC    = 86400.d0 * 365.
         ! Scale factor so that fertilizer emission = 1.8 Tg N/yr
         ! (Stehfest and Bouwman, 2006)
         ! before canopy reduction
         REAL*8, PARAMETER :: FERT_SCALE_EPIC = 0.0068 
         ! Value calculated by running the 2x2.5 GEOS-Chem model
         ! (J.D. Maasakkers)
         EPICRESERVOIRN = EPICN 
         EPICRESERVOIRN = EPICRESERVOIRN / SECPERYEAR_EPIC * FERT_SCALE_EPIC


         RETURN

         END FUNCTION EPICRESERVOIRN
C --------------------------------------------------------------------------------------------------------------------     
         SUBROUTINE GET_NDEPRES( TSTEP, NDEPRES, TAU_SEC, C, R )
C Get the deposition rate of the appropriate species for the appropriate timestep, add to reservoir and decay. 
C Return reservoir amount.          
         USE UTILIO_DEFN
         
         IMPLICIT NONE
C Function arguments:

         INTEGER, INTENT( IN )  :: TSTEP( 3 )        ! time step vector (HHMMSS)
         INTEGER, INTENT( IN )  :: C
         INTEGER, INTENT( IN )  :: R
         REAL*8,  INTENT( IN )  :: TAU_SEC
         REAL,    INTENT( INOUT ) :: NDEPRES 
         
C Local Variables
         INTEGER, SAVE :: LOGDEV

         CHARACTER( 256 ) :: MESG            ! message buffer
         CHARACTER( 16 )  :: PNAME = 'GET_NDEPRES'  ! procedure name
         REAL*8  :: C1 ! a factor   
         REAL*8  :: C2  ! another one
         REAL*8  :: TS_SEC ! time step in seconds
         real NDEPTEMP
C           check for negatives
          IF( NDEPRES < 0.0 ) THEN
          MESG = 'NDEPRES negative'
          Write(*,*) 'In GET_NDEPRES:'
          Write(*,*) 'NDEPRES negative', NDEPTEMP, NDEPRES,' ', NDEPRATE( C,R )
          write(*,*) 'TS, TAU, C1, C2:', TS_SEC, TAU_SEC,C1,C2
          CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
          ELSE IF( NDEPRATE(C,R) < 0.0 ) THEN
          MESG = 'NDEPRATE negative'
          Write(*,*) 'In GET_NDEPRES:'
          Write(*,*) 'NDEPRATE negative', NDEPRES,' ', NDEPRATE( C,R )
          CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
          write(*,*) logdev, 'logdev'
          END IF
          
         ! takes the NDEPRATE and uses it to update NDEPRES before clearing it. 
            
         !Do mass balance (see Intro to Atm Chem Chap. 3)
         !m(t) = m(0) * exp(-t/tau) + Source * tau * (1 - exp(-t/tau))
         TS_SEC = TIME2SEC(TSTEP(2))
         C1 = EXP( - TS_SEC / TAU_SEC)
         C2 = 1.d0 - C1
               NDEPTEMP = NDEPRES
               NDEPRES = NDEPRES*C1+NDEPRATE( C,R )*TAU_SEC*C2
C           check for negatives
          IF( NDEPRES < 0.0 ) THEN
          MESG = 'negative'
          Write(*,*) 'In GET_NDEPRES:'
          Write(*,*) 'NDEPRES negative', NDEPTEMP, NDEPRES,' ', NDEPRATE( C,R )
          write(*,*) 'TS, TAU, C1, C2:', TS_SEC, TAU_SEC,C1,C2
          CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
          ELSE IF( NDEPRATE(C,R) < 0.0 ) THEN
          MESG = 'negative'
          Write(*,*) 'In GET_NDEPRES:'
          Write(*,*) 'NDEPRATE negative', NDEPRES,' ', NDEPRATE( C,R )
          CALL M3EXIT( PNAME, 0, 0, MESG, XSTAT2 )
          write(*,*) logdev, 'logdev'
          END IF
         
         ! clear NDEPRATE for use during next time step
         
         NDEPRATE( C,R ) = 0.0 
          
         RETURN
         
         END SUBROUTINE GET_NDEPRES
         
         SUBROUTINE GET_N_DEP( SPEC,DEP,C,R )
            USE UTILIO_DEFN       
            IMPLICIT NONE
            CHARACTER( 256 ) :: MESG            ! message buffer
            CHARACTER( 8 ), INTENT( IN ) :: SPEC  !  dep species
            REAL,      INTENT( IN ) :: DEP !  deposition rate in kg/ha/s 
            INTEGER,   INTENT( IN ) :: C
            INTEGER,   INTENT( IN ) :: R
            REAL, PARAMETER :: HAOM2   = 1.0e-4 ! ha/m^2 conversion
            REAL, PARAMETER :: MWNH3   = 17.031 ! molecular weight of NH3
            REAL, PARAMETER :: MWNH4   = 18.039 ! molecular weight of NH4
            REAL, PARAMETER :: MWHNO3  = 63.013 ! molecular weight of HNO3
            REAL, PARAMETER :: MWNO3   = 62.005 ! molecular weight of NO3
            REAL, PARAMETER :: MWNO2   = 46.006 ! molecular weight of NO2
            REAL, PARAMETER :: MWPAN   = 121.05 ! molecular weight of Peroxyacyl nitrate
            REAL, PARAMETER :: MWN     = 14.007 ! molecular weight of Nitrogen
            REAL, PARAMETER :: NGOKG   = 1.0e12 ! ng/kg conversion
            
            ! takes Kg/hectare/s and converts to ng N / m^2/s

            IF( INDEX(TRIM( SPEC ), 'NH3') .NE. 0 ) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWNH3 
            ELSE IF( INDEX(TRIM( SPEC ), 'NH4') .NE. 0 ) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWNH4
            ELSE IF( INDEX(TRIM( SPEC ), 'HNO3') .NE. 0 ) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWHNO3
            ELSE IF( INDEX(TRIM( SPEC ), 'NO3') .NE. 0) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWNO3
            ELSE IF( INDEX(TRIM( SPEC ), 'NO2') .NE. 0 ) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWNO2
            ELSE IF( INDEX(TRIM( SPEC ), 'PAN') .NE. 0 ) THEN
               NDEPRATE( C,R ) = NDEPRATE( C,R ) + DEP*HAOM2*NGOKG*MWN/MWPAN
            Else
!               MESG = 'Invalid Species Name in Get_N_Dep: "' // SPEC // '"'
!               CALL M3EXIT( PNAME, JDATE, JTIME, MESG, XSTAT2 )
            END IF
            
            IF( (DEP<0.0) .OR. (NDEPRATE( C,R )<0.0) ) then
            Write(*,*) 'DEP or sum negative',DEP, ' ',NDEPRATE( C,R)
            MESG = 'negative'
               CALL M3EXIT( 'GET_N_DEP', 0, 0, MESG, XSTAT2 )
            END if

         RETURN

         
         END SUBROUTINE GET_N_DEP  
      END MODULE BDSNP_MOD
      
